import Cocoa

struct PointI {
  var x: Int
  var y: Int
  
  static func ==(lhs: PointI, rhs: PointI) -> Bool {
    return lhs.x == rhs.x && lhs.y == rhs.y
  }
  
  var neighbors : [PointI] {
    get {
      var neighbors = [PointI]()
    
      for x in self.x - 1...self.x + 1 {
        for y in self.y - 1...self.y + 1 {
          if x == self.x && y == self.y {
            continue
          }

          neighbors.append(PointI(x: x, y: y))
        }
      }
      
      return neighbors
    }
  }
}

class Node : CustomStringConvertible {
  var position: PointI
  var fScore: Float
  var gScore: Float
  var parent: Node? = nil
  
  init(position: PointI, fScore: Float, gScore: Float, parent: Node?) {
    self.position = position
    self.fScore = fScore
    self.gScore = gScore
    self.parent = parent
  }
  
  var description: String {
    return "(position: \(position), fScore: \(fScore), gScore: \(gScore))"
  }
}

struct Map {
  var tiles: [[Tile]]
}

protocol PathFinderDelegate {
  func isValidPosition(position: PointI) -> Bool
  func isDestination(position: PointI) -> Bool
  func movementCost(from: PointI, to: PointI) -> Float
  func heuristicCost(from: PointI) -> Float
}

extension PathFinderDelegate {
  func simpleIsValidPosition(map: Map, position: PointI) -> Bool {
    if position.x < 0 || position.x >= map.tiles.count {
      return false
    }
    
    if position.y < 0 || position.y >= map.tiles[position.x].count {
      return false
    }
    
    return !map.tiles[position.x][position.y].obstructed
  }

  func simpleMovementCost(from: PointI, to: PointI) -> Float {
    // must be a neighbouring position
    precondition(to.x >= from.x - 1 && to.x <= from.x + 1)
    precondition(to.y >= from.y - 1 && to.y <= from.y + 1)
    
    if to.x == from.x || to.y == from.y {
      return 1
    }
    
    return 1.5
  }
}

class MoveToPathFinderDelegate : PathFinderDelegate {
  let map: Map
  let destination: PointI
  
  init(map: Map, destination: PointI) {
    self.map = map
    self.destination = destination
  }
  
  func isValidPosition(position: PointI) -> Bool {
    return simpleIsValidPosition(map: map, position: position)
  }
  
  func isDestination(position: PointI) -> Bool {
    return position == self.destination
  }
  
  func movementCost(from: PointI, to: PointI) -> Float {
    return simpleMovementCost(from: from, to: to)
  }
  
  func heuristicCost(from: PointI) -> Float {
    return Float(labs(destination.x - from.x) + labs(destination.y - from.y))
  }
}

class MoveAwayPathFinderDelegate : PathFinderDelegate {
  let map: Map
  let units: [Unit]
  
  init(map: Map, units: [Unit]) {
    self.map = map
    self.units = units
  }
  
  func isValidPosition(position: PointI) -> Bool {
    if !simpleIsValidPosition(map: map, position: position) {
      return false
    }
  
    for unit in units {
      if unit.position == position {
        return false
      }
    }
    
    return true
  }
  
  func isDestination(position: PointI) -> Bool {
    for unit in units {
      if unit.path.contains(where: { $0 == position }) {
        return false
      }
    }
    
    return true
  }
  
  func movementCost(from: PointI, to: PointI) -> Float {
    return simpleMovementCost(from: from, to: to)
  }
  
  func heuristicCost(from: PointI) -> Float {
    return 0
  }
}

struct Tile {
  var obstructed: Bool
}

protocol PathFinderSet {
  func insert(node: Node)
  func remove(node: Node)
  func find(position: PointI) -> Node?
  
  var nodeWithLowestFScore : Node? { get }
  var count: Int { get }
}

class SimplePathFinderSet : PathFinderSet {
  private var nodes = [Node]()
  
  func insert(node: Node) {
    var insertIndex = nodes.endIndex
  
    for (i, existingNode) in nodes.enumerated().reversed() {
      if existingNode.fScore < node.fScore {
        insertIndex = i
      } else {
        break
      }
    }
  
    nodes.insert(node, at: insertIndex)
  }
  
  func remove(node: Node) {
    if let index = nodes.index(where: { $0 === node }) {
      nodes.remove(at: index)
    }
  }
  
  func find(position: PointI) -> Node? {
    guard let index = nodes.index(where: { $0.position == position }) else {
      return nil
    }
    
    return nodes[index];
  }
  
  var nodeWithLowestFScore : Node? {
    get {
      return nodes.last
    }
  }
  
  var count : Int {
    get {
      return nodes.count
    }
  }
}

enum UnitState {
  case Idle
  case Moving
  case Attacking
}

protocol Unit {
  var position: PointI { get }
  var path: [PointI] { get }
  var state: UnitState { get }
}

class PathFinder {
  var delegate: PathFinderDelegate

  init(delegate: PathFinderDelegate) {
    self.delegate = delegate
  }

  func findPath(origin: PointI) -> [PointI] {
    let open = SimplePathFinderSet()
    let closed = SimplePathFinderSet()
  
    open.insert(node: Node(position: origin, fScore: delegate.heuristicCost(from: origin), gScore: 0, parent: nil))
  
    while open.count != 0 {
      guard let node = open.nodeWithLowestFScore else { return [] }
      print("Lowest score node: \(node)")
      if delegate.isDestination(position: node.position) { return makePath(node) }
      
      open.remove(node: node)
      closed.insert(node: node)

      for neighbourPosition in node.position.neighbors {
        if !delegate.isValidPosition(position: neighbourPosition) { continue }
        if closed.find(position: neighbourPosition) != nil { continue }
        
        print("Testing \(neighbourPosition)")
        
        let g = node.gScore + delegate.movementCost(from: node.position, to: neighbourPosition)
        
        if let existingNode = open.find(position: neighbourPosition) {
          if g >= existingNode.gScore { continue }
          open.remove(node: existingNode)
        }
        
        let f = g + delegate.heuristicCost(from: neighbourPosition)
        
        print("Inserting \(Node(position: neighbourPosition, fScore: f, gScore: g, parent: node))")
        
        open.insert(node: Node(position: neighbourPosition, fScore: f, gScore: g, parent: node))
      }
    }
    
    return []
  }
  
  func makePath(_ node: Node) -> [PointI] {
    var path = [PointI]()
    
    path.append(node.position)
      
    var currentNode: Node? = node
      
    while currentNode != nil {
      path.append(currentNode!.position)
      currentNode = currentNode?.parent
    }
      
    return path.reversed()
  }
}

var tiles: [[Tile]] = [[Tile]]()

for x in 0...9 {
  let column = [Tile](repeating:Tile(obstructed: false), count:6)
  tiles.append(column)
}

tiles[5][2].obstructed = true
tiles[6][2].obstructed = true
tiles[7][2].obstructed = true
tiles[7][3].obstructed = true
tiles[5][4].obstructed = true
tiles[6][4].obstructed = true
tiles[7][4].obstructed = true

let map = Map(tiles: tiles)

let delegate = MoveToPathFinderDelegate(map: map, destination: PointI(x: 8, y: 3))

let pathFinder = PathFinder(delegate: delegate)

print(pathFinder.findPath(origin: PointI(x: 4, y: 3)))
